package com.nextbiometrics.devices;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothSocket;
import android.util.Log;
import com.sun.jna.Pointer;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;

public class BluetoothCommInterface extends NBUCommInterface {
    /* 26 */ public final String TAG = "CommIface";

    /* 29 */ private static final UUID BT_WELL_KNOWN_UUID = UUID.fromString("00001101-0000-1000-8000-00805F9B34FB");
    private BluetoothAdapter btAdapter;
    private Context context;

    protected void initializeHandler() {
        /* 81 */ this.btAdapter = BluetoothAdapter.getDefaultAdapter();
        /* 82 */ this.context = null;

        /* 84 */ if (this.btAdapter == null) {
            /* 86 */ Log.i("CommIface", "initializeHandler() -> NBUERROR_COMM_INIT_FAILED (Bluetooth not supported)");
            /* 87 */ throw new NBUException(1043);
        }
        /* 89 */ Log.i("CommIface", "initializeHandler() -> OK");
    }

    protected void terminateHandler() {
        /* 100 */ this.btAdapter = null;
        /* 101 */ this.context = null;
        /* 102 */ Log.i("CommIface", "terminateHandler() -> OK");
    }

    protected Pointer openHandler(String devParams, Pointer enumHandle, int devNumber) {
        /* 119 */ if ((this.btAdapter == null) || (!this.btAdapter.isEnabled())) {
            /* 120 */ Log.i("CommIface", String.format("openHandler(%s) -> NBUERROR_COMM_INIT_FAILED (BT not enabled)",
                    new Object[] { devParams }));
            /* 121 */ throw new NBUException(1043);
        }
        /* 123 */ if (this.context != null) {
            /* 125 */ closeSocket(this.context);
            /* 126 */ this.context = null;
        }

        /* 130 */ Set pairedDevices = this.btAdapter.getBondedDevices();
        /* 131 */ if (pairedDevices.size() <= 0) {
            /* 132 */ Log.i("CommIface",
                    String.format("openHandler(%s) -> NBUERROR_COMM_NO_DEVICE", new Object[] { devParams }));
            /* 133 */ throw new NBUException(1035);
        }
        /* 135 */ BluetoothDevice btDevice = null;
        /* 136 */ BluetoothSocket btSocket = null;
        /* 137 */ InputStream is = null;
        /* 138 */ OutputStream os = null;

        /* 140 */ Iterator itr = pairedDevices.iterator();
        /* 141 */ while (itr.hasNext()) {
            /* 142 */ BluetoothDevice dev = (BluetoothDevice) itr.next();
            /* 143 */ if (dev.getAddress().equals(devParams)) {
                /* 144 */ btDevice = dev;
                /* 145 */ break;
            }
        }
        /* 148 */ if (btDevice == null) {
            /* 149 */ throw new NBUException(1035);
        }
        try {
            /* 153 */ btSocket = btDevice.createRfcommSocketToServiceRecord(BT_WELL_KNOWN_UUID);
            /* 154 */ this.btAdapter.cancelDiscovery();
            /* 155 */ btSocket.connect();
            /* 156 */ is = btSocket.getInputStream();
            /* 157 */ os = btSocket.getOutputStream();
        } catch (IOException e) {
            try {
                /* 160 */ if (btSocket != null)
                    btSocket.close();
            } catch (IOException localIOException1) {
            }
            /* 164 */ Log.i("CommIface",
                    String.format("openHandler(%s) -> NBUERROR_COMM_OS_ERROR", new Object[] { devParams }));
            /* 165 */ throw new NBUException(1032);
        }

        /* 169 */ this.context = new Context();
        /* 170 */ this.context.btDevice = btDevice;
        /* 171 */ this.context.btSocket = btSocket;
        /* 172 */ this.context.is = is;
        /* 173 */ this.context.os = os;

        /* 175 */ Log.i("CommIface", String.format("openHandler(%s) -> OK", new Object[] { devParams }));
        /* 176 */ return Pointer.NULL;
    }

    protected void closeHandler() {
        /* 184 */ if (this.context != null) {
            /* 185 */ closeSocket(this.context);
            /* 186 */ this.context = null;
        }
        /* 188 */ Log.i("CommIface", "closeHandler() -> OK");
    }

    protected void reopenHandler() {
        /* 203 */ openHandler(null, null, 0);
        /* 204 */ Log.i("CommIface", "reopenHandler() -> OK");
    }

    protected void setTimeoutHandler(int timeoutMs) {
        /* 216 */ if (this.context == null) {
            /* 217 */ Log.i("CommIface", String.format("setTimeoutHandler(%d) -> NBUERROR_GENERAL",
                    new Object[] { Integer.valueOf(timeoutMs) }));
            /* 218 */ throw new NBUException(128);
        }
        /* 220 */ if (timeoutMs <= 0) {
            /* 221 */ Log.i("CommIface", String.format("setTimeoutHandler(%d) -> NBUERROR_BAD_PARAM",
                    new Object[] { Integer.valueOf(timeoutMs) }));
            /* 222 */ throw new NBUException(18);
        }
        /* 224 */ this.context.timoutMs = timeoutMs;
        /* 225 */ Log.i("CommIface",
                String.format("setTimeoutHandler(%d) -> OK", new Object[] { Integer.valueOf(timeoutMs) }));
    }

    protected void waitForPacketStartHandler() {
        /* 242 */ if (this.context == null) {
            /* 243 */ Log.i("CommIface",
                    String.format("waitForPacketStartHandler() -> NBUERROR_GENERAL", new Object[0]));
            /* 244 */ throw new NBUException(128);
        }
        while (true) {
            /* 248 */ int b = readByte(this.context);
            /* 249 */ if (b < 0) {
                /* 250 */ Log.i("CommIface",
                        String.format("waitForPacketStartHandler() -> NBUERROR_COMM_TIMEOUT", new Object[0]));
                /* 251 */ throw new NBUException(1033);
                /* 252 */ }
            if (b == 27) {
                /* 255 */ b = readByte(this.context);
                /* 256 */ if (b < 0) {
                    /* 257 */ Log.i("CommIface",
                            String.format("waitForPacketStartHandler() -> NBUERROR_COMM_TIMEOUT", new Object[0]));
                    /* 258 */ throw new NBUException(1033);
                    /* 259 */ }
                if (b == 2)
                    break;
            }
        }
        /* 262 */ Log.i("CommIface", String.format("waitForPacketStartHandler() -> OK", new Object[0]));
    }

    protected byte receiveByteHandler() {
        /* 277 */ int value = 0;

        /* 279 */ if (this.context == null) {
            /* 280 */ Log.i("CommIface", String.format("receiveByteHandler() -> NBUERROR_GENERAL", new Object[0]));
            /* 281 */ throw new NBUException(128);
        }

        /* 284 */ value = readByte(this.context);
        /* 285 */ if (value < 0) {
            /* 286 */ Log.i("CommIface", String.format("receiveByteHandler() -> NBUERROR_COMM_TIMEOUT", new Object[0]));
            /* 287 */ throw new NBUException(1033);
            /* 288 */ }
        if (value == 27) {
            /* 290 */ value = readByte(this.context);
            /* 291 */ if (value < 0) {
                /* 292 */ Log.i("CommIface",
                        String.format("receiveByteHandler() -> NBUERROR_COMM_TIMEOUT", new Object[0]));
                /* 293 */ throw new NBUException(1033);
                /* 294 */ }
            if (value == 1) {
                /* 295 */ value = 27;
            } else {
                /* 296 */ if (value == 2) {
                    /* 298 */ Log.i("CommIface",
                            String.format("receiveByteHandler() -> NBUERROR_COMM_START_PACKET", new Object[0]));
                    /* 299 */ throw new NBUException(1027);
                }

                /* 302 */ Log.i("CommIface",
                        String.format("receiveByteHandler() -> NBUERROR_COMM_DATA_ERROR", new Object[0]));
                /* 303 */ throw new NBUException(1028);
            }

        }

        /* 308 */ return (byte) (value & 0xFF);
    }

    protected void setRemainingSizeHandler(int receiveDataSize) {
        /* 327 */ receiveDataSize -= this.context.ibytes;
        /* 328 */ this.context.irequested = (receiveDataSize > 0 ? receiveDataSize : 0);
        /* 329 */ Log.i("CommIface",
                String.format("setRemainingSizeHandler(%d) -> OK", new Object[] { Integer.valueOf(receiveDataSize) }));
    }

    protected void startPacketSendHandler(int packetSize) {
        /* 348 */ if (this.context == null) {
            /* 349 */ Log.i("CommIface", String.format("startPacketSendHandler(%d) -> NBUERROR_GENERAL",
                    new Object[] { Integer.valueOf(packetSize) }));
            /* 350 */ throw new NBUException(128);
        }

        try {
            /* 355 */ this.context.ibytes = 0;
            /* 356 */ this.context.iindex = 0;
            /* 357 */ this.context.irequested = 0;
            /* 358 */ while (this.context.is.available() > 0)
                /* 359 */ this.context.is.read();
        } catch (IOException e) {
            /* 361 */ Log.i("CommIface", String.format("startPacketSendHandler(%d) -> NBUERROR_COMM_OS_ERROR",
                    new Object[] { Integer.valueOf(packetSize) }));
            /* 362 */ throw new NBUException(1032);
        }

        /* 366 */ if ((this.context.obuffer == null) || (this.context.obuffer.length < 2 * packetSize + 2)) {
            /* 367 */ this.context.obuffer = new byte[2 * packetSize + 2];
        }
        /* 369 */ this.context.obytes = 0;
        /* 370 */ this.context.obuffer[(this.context.obytes++)] = 27;
        /* 371 */ this.context.obuffer[(this.context.obytes++)] = 2;
        /* 372 */ Log.i("CommIface",
                String.format("startPacketSendHandler(%d) -> OK", new Object[] { Integer.valueOf(packetSize) }));
    }

    protected void sendByteHandler(byte value, boolean lastByte) {
        /* 391 */ if (this.context == null) {
            /* 392 */ Log.i("CommIface", String.format("sendByteHandler(0x%02x, %s) -> NBUERROR_GENERAL",
                    new Object[] { Byte.valueOf(value), lastByte ? "true" : "false" }));
            /* 393 */ throw new NBUException(128);
        }

        /* 396 */ if (value == 27) {
            /* 397 */ if (this.context.obuffer.length - this.context.obytes < 2) {
                /* 398 */ Log.i("CommIface",
                        String.format("sendByteHandler(0x%02x, %s) -> NBUERROR_COMM_BAD_DATA_LENGTH",
                                new Object[] { Byte.valueOf(value), lastByte ? "true" : "false" }));
                /* 399 */ throw new NBUException(1039);
            }
            /* 401 */ this.context.obuffer[(this.context.obytes++)] = 27;
            /* 402 */ this.context.obuffer[(this.context.obytes++)] = 1;
        } else {
            /* 404 */ if (this.context.obuffer.length - this.context.obytes < 1) {
                /* 405 */ Log.i("CommIface",
                        String.format("sendByteHandler(0x%02x, %s) -> NBUERROR_COMM_BAD_DATA_LENGTH",
                                new Object[] { Byte.valueOf(value), lastByte ? "true" : "false" }));
                /* 406 */ throw new NBUException(1039);
            }
            /* 408 */ this.context.obuffer[(this.context.obytes++)] = value;
        }

        /* 412 */ if (lastByte) {
            /* 413 */ boolean success = writeBytesWithTimeout(this.context, this.context.obuffer, this.context.obytes);
            /* 414 */ Log.i("CommIface", String.format("sendByteHandler flushing buffer %d bytes -> %s",
                    new Object[] { Integer.valueOf(this.context.obytes), success ? "OK" : "NBUERROR_COMM_TIMEOUT" }));
            /* 415 */ this.context.obytes = 0;
            /* 416 */ if (!success)
                throw new NBUException(1033);
        }
    }

    protected void closeSocket(Context context) {
        /* 425 */ if ((context != null) && (context.btSocket != null))
            try {
                /* 427 */ context.btSocket.close();
            } catch (IOException localIOException) {
            }
    }

    protected int readByte(Context context) {
        /* 436 */ int result = 0;
        /* 437 */ if (context.iindex < context.ibytes) {
            /* 439 */ result = context.ibuffer[(context.iindex++)] & 0xFF;
        }
        /* 442 */ else if (context.irequested > 0) {
            /* 444 */ int length = context.irequested;
            /* 445 */ context.irequested = 0;
            /* 446 */ context.ibytes = 0;
            /* 447 */ context.iindex = 0;
            /* 448 */ if ((context.ibuffer == null) || (context.ibuffer.length < length)) {
                /* 449 */ context.ibuffer = new byte[length];
            }

            /* 453 */ boolean success = readBytesWithTimeout(context, context.ibuffer, length);
            /* 454 */ if (!success) {
                /* 455 */ result = -1;
                /* 456 */ Log.i("CommIface", String.format("readByte() -> %d, attempt to read %d bytes failed",
                        new Object[] { Integer.valueOf(result), Integer.valueOf(length) }));
            } else {
                /* 459 */ context.ibytes = length;
                /* 460 */ result = context.ibuffer[(context.iindex++)];
            }

        } else {
            /* 466 */ result = readByteWithTimeout(context);
        }

        /* 469 */ return result;
    }

    protected int readByteWithTimeout(Context context) {
        /* 495 */ TimerTask task = new TimerTask() {
            /* 476 */ BluetoothCommInterface.Context context = null;

            public void run() {
                try {
                    /* 485 */ if (this.context.btSocket != null)
                        /* 486 */ this.context.btSocket.close();
                } catch (IOException localIOException) {
                }
            }
        };
        /* 496 */ Timer timer = new Timer();
        /* 497 */ timer.schedule(task, context.timoutMs);
        int response;
        try {
            /* 499 */ response = context.is.read();
        } catch (IOException e) {
            /* 501 */ response = -1;
        }
        /* 503 */ timer.cancel();
        /* 504 */ return response;
    }

    protected boolean readBytesWithTimeout(Context context, byte[] bytes, int nrBytes) {
        /* 529 */ boolean response = true;
        /* 530 */ int index = 0;
        /* 531 */ while ((response) && (index < nrBytes)) {
            /* 532 */ TimerTask task = new TimerTask() {
                /* 511 */ BluetoothCommInterface.Context context = null;

                public void run() {
                    try {
                        /* 520 */ if (this.context.btSocket != null)
                            /* 521 */ this.context.btSocket.close();
                    } catch (IOException localIOException) {
                    }
                }
            };
            /* 533 */ Timer timer = new Timer();
            /* 534 */ timer.schedule(task, context.timoutMs + (nrBytes - index) * 10000 / context.baudrate);
            /*     */ try {
                /* 536 */ int read = context.is.read(bytes, index, nrBytes - index);
                /* 537 */ if (read <= 0)
                    /* 538 */ response = false;
                else
                    /* 540 */ index += read;
            } catch (IOException e) {
                /* 543 */ response = false;
            }
            /* 545 */ timer.cancel();
        }

        /* 548 */ return response;
    }

    protected boolean writeByteWithTimeout(Context context, int value) {
        /* 573 */ boolean result = true;
        /* 574 */ TimerTask task = new TimerTask() {
            /* 555 */ BluetoothCommInterface.Context context = null;

            public void run() {
                try {
                    /* 564 */ if (this.context.btSocket != null)
                        /* 565 */ this.context.btSocket.close();
                } catch (IOException localIOException) {
                }
            }
        };
        /* 575 */ Timer timer = new Timer();
        /* 576 */ timer.schedule(task, context.timoutMs);
        try {
            /* 578 */ context.os.write(value);
        } catch (IOException e) {
            /* 580 */ result = false;
        }
        /* 582 */ timer.cancel();
        /* 583 */ return result;
    }

    protected boolean writeBytesWithTimeout(Context context, byte[] bytes, int nrBytes) {
        /* 608 */ boolean result = true;
        /* 609 */ TimerTask task = new TimerTask() {
            /* 590 */ BluetoothCommInterface.Context context = null;

            public void run() {
                try {
                    /* 599 */ if (this.context.btSocket != null)
                        /* 600 */ this.context.btSocket.close();
                } catch (IOException localIOException) {
                }
            }
        };
        /* 610 */ Timer timer = new Timer();
        /* 611 */ timer.schedule(task, context.timoutMs + nrBytes * 10000 / context.baudrate);
        try {
            /* 613 */ context.os.write(bytes, 0, nrBytes);
        } catch (IOException e) {
            /* 615 */ result = false;
        }
        /* 617 */ timer.cancel();
        /* 618 */ return result;
    }

    public class Context {
        /* 36 */ public BluetoothDevice btDevice = null;
        /* 37 */ public BluetoothSocket btSocket = null;
        /* 38 */ public InputStream is = null;
        /* 39 */ public OutputStream os = null;
        /* 40 */ public int timoutMs = 2000;
        /* 41 */ public int baudrate = 115200;
        /* 42 */ public byte[] ibuffer = null;
        /* 43 */ public byte[] obuffer = null;
        /* 44 */ public int ibytes = 0;
        /* 45 */ public int obytes = 0;
        /* 46 */ public int iindex = 0;
        /* 47 */ public int irequested = 0;

        public Context() {
        }
    }
}
