package com.nextbiometrics.devices;

import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbDeviceConnection;
import android.hardware.usb.UsbManager;
import android.util.Log;
import com.nextbiometrics.system.NBErrors;
import com.nextbiometrics.system.jna.NBSizeType;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

class AndroidDeviceEnumHelper implements NBDevicesDeviceEnumHelper.IDeviceEnumHelper {
    private static final int NEXT_VENDOR_ID = 10637;
    private static final String ACTION_USB_PERMISSION = "com.nextbiometrics.devices.USB_PERMISSION";
    private static final String TAG = "AndroidDeviceEnumHelper";
    /* 82 */ private Context context = null;
    /* 83 */ private UsbManager usbManager = null;
    /* 84 */ private PendingIntent permissionIntent = null;
    /* 85 */ private NBDevicesDeviceEnumHelper.NBDevicesReenumerateDevicesProc pReenumerateDevicesProc = null;
    /* 86 */ private NBSizeType pReenumerateParam = null;

    /* 88 */ private Map<String, NBDeviceUsbEx> devicesMap = Collections.synchronizedMap(new HashMap());
    private boolean isInitDone;
    /* 92 */ private final BroadcastReceiver usbReceiver = new BroadcastReceiver() {
        public void onReceive(Context context, Intent intent) {
            try {
                /* 95 */ boolean requestEnumerate = false;
                /* 96 */ String action = intent.getAction();
                /* 97 */ synchronized (this) {
                    /* 98 */ if ("com.nextbiometrics.devices.USB_PERMISSION".equals(action)) {
                        /* 99 */ UsbDevice usbDevice = (UsbDevice) intent.getParcelableExtra("device");
                        /* 100 */ if (usbDevice != null) {
                            /* 101 */ String deviceId = usbDevice.getDeviceName();
                            /* 102 */ if (intent.getBooleanExtra("permission", false)) {
                                /* 103 */ if (AndroidDeviceEnumHelper.this.devicesMap.containsKey(deviceId)) {
                                    /* 104 */ AndroidDeviceEnumHelper.this.addDevice(usbDevice);
                                }
                            }
                            /* 107 */ else if (AndroidDeviceEnumHelper.this.devicesMap.containsKey(deviceId)) {
                                /* 108 */ AndroidDeviceEnumHelper.NBDeviceUsbEx deviceUsbEx = (AndroidDeviceEnumHelper.NBDeviceUsbEx) AndroidDeviceEnumHelper.this.devicesMap
                                        .get(deviceId);
                                /* 109 */ if (deviceUsbEx != null) {
                                    /* 110 */ AndroidDeviceEnumHelper.this.deleteDevice(deviceUsbEx);
                                }
                            }
                            /* 113 */ requestEnumerate = true;
                        }
                    }
                    /* 116 */ if (("android.hardware.usb.action.USB_ACCESSORY_ATTACHED".equals(action)) ||
                    /* 117 */ ("android.hardware.usb.action.USB_ACCESSORY_ATTACHED"/* 117 */ .equals(action)) ||
                    /* 118 */ ("android.hardware.usb.action.USB_DEVICE_ATTACHED"/* 118 */ .equals(action)) ||
                    /* 119 */ ("android.hardware.usb.action.USB_DEVICE_DETACHED"/* 119 */ .equals(action))) {
                        /* 120 */ requestEnumerate = true;
                    }
                }
                /* 123 */ if ((requestEnumerate) && (AndroidDeviceEnumHelper.this.pReenumerateDevicesProc != null))
                    /* 124 */ NBErrors.requireNoErr(AndroidDeviceEnumHelper.this.pReenumerateDevicesProc
                            .invoke(AndroidDeviceEnumHelper.this.pReenumerateParam));
            } catch (Throwable ex) {
                /* 128 */ Log.e("AndroidDeviceEnumHelper", "Error in BroadcastReceiver", ex);
            }
        }
        /* 92 */ };

    AndroidDeviceEnumHelper(Object context) {
        /* 138 */ this.context = ((Context) context);
    }

    public void initialize(NBDevicesDeviceEnumHelper.NBDevicesReenumerateDevicesProc pReenumerateDevicesProc,
            NBSizeType pParam) {
        /* 147 */ synchronized (this) {
            /* 148 */ if (isInitialized())
                return;
            /* 149 */ this.pReenumerateDevicesProc = pReenumerateDevicesProc;
            /* 150 */ this.pReenumerateParam = pParam;
            /* 151 */ this.usbManager = ((UsbManager) this.context.getSystemService("usb"));
            /* 152 */ this.permissionIntent = PendingIntent.getBroadcast(this.context, 0,
                    new Intent("com.nextbiometrics.devices.USB_PERMISSION"), 0);
            /* 153 */ IntentFilter filter = new IntentFilter();
            /* 154 */ filter.addAction("com.nextbiometrics.devices.USB_PERMISSION");
            /* 155 */ filter.addAction("android.hardware.usb.action.USB_ACCESSORY_ATTACHED");
            /* 156 */ filter.addAction("android.hardware.usb.action.USB_ACCESSORY_DETACHED");
            /* 157 */ filter.addAction("android.hardware.usb.action.USB_DEVICE_ATTACHED");
            /* 158 */ filter.addAction("android.hardware.usb.action.USB_DEVICE_DETACHED");
            /* 159 */ this.context.registerReceiver(this.usbReceiver, filter);
            /* 160 */ this.isInitDone = true;
        }
    }

    public boolean isInitialized() {
        /* 165 */ return this.isInitDone;
    }

    public void terminate() {
        /* 170 */ synchronized (this) {
            /* 171 */ if (!isInitialized())
                return;
            /* 172 */ this.context.unregisterReceiver(this.usbReceiver);
            /* 173 */ clearDevices();
            /* 174 */ this.usbManager = null;
            /* 175 */ this.permissionIntent = null;
        }
    }

    public void update(NBDevice[] devices, NBDevicesDeviceEnumHelper.NBDevicesNewDeviceProc pNewDeviceProc,
            NBDevicesDeviceEnumHelper.NBDevicesFoundDeviceProc pFoundDeviceProc, NBSizeType pParam) {
        /* 181 */ synchronized (this) {
            /* 183 */ clearDisconnectedDevices();

            /* 185 */ HashMap<String, UsbDevice> deviceList = this.usbManager.getDeviceList();
            /* 186 */ for (UsbDevice device : deviceList.values())
                /* 187 */ if ((device.getVendorId() == 10637) &&
                /* 188 */ (!NBDevice.SupportsNBUApi(device/* 188 */ .getVendorId(), device.getProductId()))) {
                    /* 191 */ String deviceId = device.getDeviceName();

                    /* 193 */ if (!this.devicesMap.containsKey(deviceId)) {
                        /* 194 */ this.devicesMap.put(device.getDeviceName(), null);
                        /* 195 */ this.usbManager.requestPermission(device, this.permissionIntent);
                    } else {
                        /* 198 */ NBDeviceUsbEx deviceUsbEx = (NBDeviceUsbEx) this.devicesMap.get(deviceId);

                        /* 200 */ if (deviceUsbEx != null) {
                            /* 203 */ NBDevice found = null;
                            /* 204 */ for (NBDevice d : devices) {
                                /* 205 */ String otherDeviceId = d.getId();

                                /* 207 */ if (deviceId.equals(otherDeviceId)) {
                                    /* 208 */ found = d;
                                    /* 209 */ break;
                                }
                            }

                            /* 213 */ if ((found != null)
                                    && (deviceUsbEx.getStatus() != NBDeviceUsbExStatus.IS_ADDED)) {
                                /* 214 */ removeDevice(deviceId);
                            }
                            /* 218 */ else if (found != null) {
                                /* 219 */ NBErrors.requireNoErr(pFoundDeviceProc.invoke(found.getHandle(), pParam));
                            } else {
                                /* 223 */ NBErrors.requireNoErr(
                                        pNewDeviceProc.invoke(deviceUsbEx.getDevice().getHandle(), pParam));
                                /* 224 */ deviceUsbEx.setStatus(NBDeviceUsbExStatus.IS_ADDED);
                            }
                        }
                    }
                }
        }
    }

    private void deleteDevice(NBDeviceUsbEx deviceUsbEx) {
        /* 236 */ if (deviceUsbEx == null)
            return;
        /* 237 */ NBDevice device = deviceUsbEx.getDevice();
        /* 238 */ if (device != null)
            device.dispose();
    }

    private void removeDevice(String deviceId) {
        /* 244 */ if (this.devicesMap.containsKey(deviceId)) {
            /* 245 */ NBDeviceUsbEx deviceUsbEx = (NBDeviceUsbEx) this.devicesMap.remove(deviceId);
            /* 246 */ deleteDevice(deviceUsbEx);
        }
    }

    private void removeDevice(NBDeviceUsbEx deviceUsbEx) {
        /* 251 */ if (deviceUsbEx != null)
            /* 252 */ removeDevice(deviceUsbEx.getId());
    }

    private void clearDevices() {
        /* 257 */ Iterator deviceUsbIterator = this.devicesMap.values().iterator();
        /* 258 */ while (deviceUsbIterator.hasNext()) {
            /* 259 */ NBDeviceUsbEx deviceUsbEx = (NBDeviceUsbEx) deviceUsbIterator.next();
            /* 260 */ if (deviceUsbEx != null) {
                /* 261 */ removeDevice(deviceUsbEx.getId());
            }
        }
        /* 264 */ this.devicesMap.clear();
    }

    private void clearDisconnectedDevices() {
        /* 268 */ Iterator deviceUsbIterator = this.devicesMap.values().iterator();
        /* 269 */ while (deviceUsbIterator.hasNext()) {
            /* 270 */ NBDeviceUsbEx deviceUsbEx = (NBDeviceUsbEx) deviceUsbIterator.next();
            /* 271 */ if (deviceUsbEx != null) {
                /* 272 */ NBDevice device = deviceUsbEx.getDevice();
                /* 273 */ NBDeviceState state = device.getState();
                /* 274 */ if (state == NBDeviceState.NOT_CONNECTED) {
                    /* 275 */ deleteDevice(deviceUsbEx);
                    /* 276 */ this.devicesMap.remove(deviceUsbEx.getId());
                }
            }
        }
    }

    private boolean addDevice(UsbDevice usbDevice) {
        /* 283 */ String deviceId = usbDevice.getDeviceName();
        /* 284 */ if (this.devicesMap.containsKey(deviceId)) {
            /* 285 */ NBDeviceUsbEx deviceUsbEx = (NBDeviceUsbEx) this.devicesMap.get(deviceId);
            /* 286 */ if ((deviceUsbEx == null) || (deviceUsbEx.getStatus() == NBDeviceUsbExStatus.NONE))
                /* 287 */ removeDevice(deviceId);
            else {
                /* 289 */ return false;
            }
        }
        /* 292 */ UsbDeviceConnection usbDeviceConnection = this.usbManager.openDevice(usbDevice);
        /* 293 */ if (usbDeviceConnection == null)
            return false;
        /* 294 */ NBDevice device = connect(usbDevice, usbDeviceConnection);
        /* 295 */ if (device == null)
            return false;
        /* 296 */ this.devicesMap.put(deviceId,
                new NBDeviceUsbEx(deviceId, usbDevice, usbDeviceConnection, device, NBDeviceUsbExStatus.IS_PROCESSED));
        /* 297 */ return true;
    }

    private NBDevice connect(UsbDevice usbDevice, UsbDeviceConnection usbDeviceConnection) {
        try {
            /* 302 */ String path = usbDevice.getDeviceName();
            /* 303 */ int fd = usbDeviceConnection.getFileDescriptor();
            /* 304 */ int vendorId = usbDevice.getVendorId();
            /* 305 */ int productId = usbDevice.getProductId();
            /* 306 */ return NBDevice.connect(path, fd, vendorId, productId, 0L);
        } catch (Throwable ex) {
            /* 309 */ Log.e("AndroidDeviceEnumHelper", "Error in connect", ex);
            /* 310 */ }
        return null;
    }

    private class NBDeviceUsbEx {
        /* 35 */ private String id = null;
        /* 36 */ private UsbDevice usbDevice = null;
        /* 37 */ private UsbDeviceConnection usbDeviceConnection = null;
        /* 38 */ private NBDevice device = null;
        /* 39 */ private AndroidDeviceEnumHelper.NBDeviceUsbExStatus status = AndroidDeviceEnumHelper.NBDeviceUsbExStatus.NONE;

        NBDeviceUsbEx(String id, UsbDevice usbDevice, UsbDeviceConnection usbDeviceConnection, NBDevice device,
                AndroidDeviceEnumHelper.NBDeviceUsbExStatus status) {
            /* 42 */ this.id = id;
            /* 43 */ this.usbDevice = usbDevice;
            /* 44 */ this.usbDeviceConnection = usbDeviceConnection;
            /* 45 */ this.device = device;
            /* 46 */ this.status = status;
        }

        public String getId() {
            /* 49 */ return this.id;
        }

        public UsbDevice getUsbDevice() {
            /* 52 */ return this.usbDevice;
        }

        public UsbDeviceConnection getUsbDeviceConnection() {
            /* 55 */ return this.usbDeviceConnection;
        }

        NBDevice getDevice() {
            /* 58 */ return this.device;
        }

        AndroidDeviceEnumHelper.NBDeviceUsbExStatus getStatus() {
            /* 62 */ return this.status;
        }

        void setStatus(AndroidDeviceEnumHelper.NBDeviceUsbExStatus value) {
            /* 66 */ this.status = value;
        }
    }

    private static enum NBDeviceUsbExStatus {
        /* 28 */ NONE, /* 29 */ IS_PROCESSED, /* 30 */ IS_ADDED;
    }
}
