package com.nextbiometrics.devices;

import com.nextbiometrics.devices.event.NBDeviceScanPreviewListener;
import com.nextbiometrics.devices.jna.NBDeviceInfoAData;
import com.nextbiometrics.devices.jna.NBDeviceInfoAData.ByReference;
import com.nextbiometrics.devices.jna.NBDeviceInfoExAData;
import com.nextbiometrics.devices.jna.NBDeviceScanFormatInfoData;
import com.nextbiometrics.devices.jna.NBDeviceScanPreviewCallback;
import com.nextbiometrics.devices.jna.NBDeviceScanPreviewCallback.NBDeviceScanPreviewProc;
import com.nextbiometrics.devices.jna.NBUCommInterfaceStruct;
import com.nextbiometrics.system.NBCallbacks;
import com.nextbiometrics.system.NBErrors;
import com.nextbiometrics.system.NBObject;
import com.nextbiometrics.system.NBVersion;
import com.nextbiometrics.system.jna.NBSizeType;
import com.nextbiometrics.system.jna.NBUInt;
import com.nextbiometrics.system.jna.NBVersionData;
import com.nextbiometrics.system.jna.ptr.NBUIntByReference;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.IntByReference;
import com.sun.jna.ptr.PointerByReference;

public final class NBDevice extends NBObject {
    public static final int SCAN_TIMEOUT_INFINITE = -1;
    /* 33 */ public static final NBDeviceFingerDetectType DEFAULT_FINGER_DETECT_TYPE = NBDeviceFingerDetectType.ENHANCED;
    public static final int DEFAULT_FINGER_DETECT_THRESHOLD = 40;
    public static final int SCAN_SKIP_FINGER_NOT_REMOVED_STATUS_FLAG = 8;
    public static final int DEVICE_CONNECT_TO_SPI_SKIP_GPIO_INIT_FLAG = 2;
    public static final int NB_DEVICE_PARAMETER_SUBTRACT_BACKGROUND = 105;
    public static final int NB_DEVICE_PARAMETER_IMAGE_TYPE = 110;
    public static final int BLOB_PARAMETER_CALIBRATION_DATA = 201;
    public static final int BLOB_PARAMETER_SET_CDK = 202;

    private static native int NBDeviceConnectA(NBDeviceInfoAData.ByReference paramByReference, int paramInt,
            PointerByReference paramPointerByReference);

    private static native int NBDeviceConnectExA(NBDeviceInfoExAData.ByReference paramByReference, int paramInt,
            PointerByReference paramPointerByReference);

    private static native int NBDeviceConnectToSpiA(String paramString, int paramInt1, int paramInt2, int paramInt3,
            PointerByReference paramPointerByReference);

    private static native int NBDeviceConnectToSpiExA(String paramString, int paramInt1, int paramInt2, int paramInt3,
            int paramInt4, PointerByReference paramPointerByReference);

    private static native int NBDeviceConnectToCustom(NBUCommInterfaceStruct paramNBUCommInterfaceStruct, int paramInt,
            PointerByReference paramPointerByReference);

    private static native int NBDeviceDestroy(Pointer paramPointer);

    private static native int NBDeviceGetState(Pointer paramPointer, IntByReference paramIntByReference);

    private static native int NBDeviceReset(Pointer paramPointer);

    private static native int NBDeviceGetType(Pointer paramPointer, IntByReference paramIntByReference);

    private static native int NBDeviceGetConnectionType(Pointer paramPointer, IntByReference paramIntByReference);

    private static native int NBDeviceGetFirmwareVersion(Pointer paramPointer,
            NBVersionData.ByReference paramByReference);

    private static native int NBDeviceGetIdA(Pointer paramPointer, byte[] paramArrayOfByte, NBUInt paramNBUInt,
            NBUIntByReference paramNBUIntByReference);

    private static native int NBDeviceGetManufacturerA(Pointer paramPointer, byte[] paramArrayOfByte,
            NBUInt paramNBUInt, NBUIntByReference paramNBUIntByReference);

    private static native int NBDeviceGetModelA(Pointer paramPointer, byte[] paramArrayOfByte, NBUInt paramNBUInt,
            NBUIntByReference paramNBUIntByReference);

    private static native int NBDeviceGetSerialNumberA(Pointer paramPointer, byte[] paramArrayOfByte,
            NBUInt paramNBUInt, NBUIntByReference paramNBUIntByReference);

    private static native int NBDeviceGetModuleSerialNumberA(Pointer paramPointer, byte[] paramArrayOfByte,
            NBUInt paramNBUInt, NBUIntByReference paramNBUIntByReference);

    private static native int NBDeviceGetProductA(Pointer paramPointer, byte[] paramArrayOfByte, NBUInt paramNBUInt,
            NBUIntByReference paramNBUIntByReference);

    private static native int NBDeviceGetSupportedScanFormats(Pointer paramPointer, int[] paramArrayOfInt,
            NBUInt paramNBUInt, NBUIntByReference paramNBUIntByReference);

    private static native int NBDeviceIsScanFormatSupported(Pointer paramPointer, int paramInt,
            IntByReference paramIntByReference);

    private static native int NBDeviceGetScanFormatInfo(Pointer paramPointer, int paramInt,
            NBDeviceScanFormatInfoData.ByReference paramByReference);

    private static native int NBDeviceGetParameter(Pointer paramPointer, int paramInt,
            IntByReference paramIntByReference);

    private static native int NBDeviceSetParameter(Pointer paramPointer, int paramInt1, int paramInt2);

    private static native int NBDeviceSetLedState(Pointer paramPointer, int paramInt);

    private static native int NBDeviceEnterStopMode(Pointer paramPointer, int paramInt1, int paramInt2);

    private static native int NBDeviceSetBlobParameter(Pointer paramPointer, int paramInt1, byte[] paramArrayOfByte,
            int paramInt2);

    private static native int NBDeviceGetBlobParameter(Pointer paramPointer, int paramInt,
            PointerByReference paramPointerByReference, IntByReference paramIntByReference);

    private static native int NBDeviceGetFingerDetectValue(Pointer paramPointer, int paramInt,
            IntByReference paramIntByReference);

    private static native int NBDeviceIsScanRunning(Pointer paramPointer, IntByReference paramIntByReference);

    private static native int NBDeviceCancelScan(Pointer paramPointer);

    private static native int NBDeviceScan(Pointer paramPointer, int paramInt1, byte[] paramArrayOfByte,
            NBSizeType paramNBSizeType, int paramInt2, IntByReference paramIntByReference);

    private static native int NBDeviceScanEx(Pointer paramPointer, int paramInt1, int paramInt2,
            NBDeviceScanPreviewCallback.NBDeviceScanPreviewProc paramNBDeviceScanPreviewProc,
            NBSizeType paramNBSizeType1, byte[] paramArrayOfByte, NBSizeType paramNBSizeType2, int paramInt3,
            IntByReference paramIntByReference);

    private static native int NBDeviceConvertImage(byte[] paramArrayOfByte, int paramInt1, int paramInt2, int paramInt3,
            int paramInt4, float paramFloat, int paramInt5, int paramInt6, PointerByReference paramPointerByReference,
            IntByReference paramIntByReference);

    private static native int NBDeviceFree(Pointer paramPointer);

    private static native int NBDeviceOpenSession(Pointer paramPointer, byte[] paramArrayOfByte1,
            NBSizeType paramNBSizeType1, byte[] paramArrayOfByte2, NBSizeType paramNBSizeType2);

    private static native int NBDeviceCloseSession(Pointer paramPointer);

    private static native int NBDeviceIsSessionOpen(Pointer paramPointer, IntByReference paramIntByReference);

    private static native int NBDeviceGetCapabilities(Pointer paramPointer, PointerByReference paramPointerByReference);

    private static native int NBDeviceSetCustomScanFormat(Pointer paramPointer, int paramInt1, int paramInt2,
            int paramInt3, int paramInt4, int paramInt5, int paramInt6);

    private static native int NBDeviceGenerateCalibrationData(Pointer paramPointer,
            PointerByReference paramPointerByReference, IntByReference paramIntByReference);

    private static native int NBDeviceSupportsNBUApi(int paramInt1, int paramInt2, IntByReference paramIntByReference);

    private static native int NBDeviceImageQuality(byte[] paramArrayOfByte, int paramInt1, int paramInt2, int paramInt3,
            int paramInt4, IntByReference paramIntByReference);

    public NBDevice(Pointer handle, boolean ownsHandle) {
        /* 136 */ super(handle, ownsHandle);
    }

    public NBDevice(Pointer handle) {
        /* 145 */ this(handle, true);
    }

    static NBDevice connect(NBDeviceInfoAData data, long flags) {
        /* 153 */ PointerByReference hHandle = new PointerByReference();
        /* 154 */ NBDeviceInfoAData.ByReference deviceInfoRef = new NBDeviceInfoAData.ByReference(data);
        /* 155 */ NBErrors.requireNoErr(NBDeviceConnectA(deviceInfoRef, (int) flags, hHandle));
        /* 156 */ return new NBDevice(hHandle.getValue());
    }

    static NBDevice connect(String path, int fileDescriptor, int vendorId, int productId, long flags) {
        /* 160 */ NBDeviceInfoExAData.ByReference psDeviceInfo = new NBDeviceInfoExAData.ByReference(path,
                fileDescriptor, vendorId, productId);
        /* 161 */ PointerByReference hHandle = new PointerByReference();
        /* 162 */ NBErrors.requireNoErr(NBDeviceConnectExA(psDeviceInfo, (int) flags, hHandle));
        /* 163 */ return new NBDevice(hHandle.getValue());
    }

    public final NBDeviceState getState() {
        /* 176 */ IntByReference value = new IntByReference();
        /* 177 */ NBErrors.requireNoErr(NBDeviceGetState(getHandle(), value));
        /* 178 */ return NBDeviceState.get(value.getValue());
    }

    public final NBDeviceType getType() {
        /* 187 */ IntByReference value = new IntByReference();
        /* 188 */ NBErrors.requireNoErr(NBDeviceGetType(getHandle(), value));
        /* 189 */ return NBDeviceType.get(value.getValue());
    }

    public final NBDeviceConnectionType getConnectionType() {
        /* 198 */ IntByReference value = new IntByReference();
        /* 199 */ NBErrors.requireNoErr(NBDeviceGetConnectionType(getHandle(), value));
        /* 200 */ return NBDeviceConnectionType.get(value.getValue());
    }

    public final NBVersion getFirmwareVersion() {
        NBVersionData.ByReference value = new NBVersionData.ByReference();
        NBErrors.requireNoErr(NBDeviceGetFirmwareVersion(getHandle(), value));
        return value.toVersion();
    }

    public final String getId() {
        NBUIntByReference value = new NBUIntByReference();
        NBErrors.requireNoErr(NBDeviceGetIdA(getHandle(), null, new NBUInt(0L), value));
        byte[] id = new byte[value.getValue().intValue() + 1];
        NBErrors.requireNoErr(NBDeviceGetIdA(getHandle(), id, new NBUInt(id.length), value));
        return Native.toString(id);
    }

    public final String getManufacturer() {
        NBUIntByReference value = new NBUIntByReference();
        NBErrors.requireNoErr(NBDeviceGetManufacturerA(getHandle(), null, new NBUInt(0L), value));
        byte[] id = new byte[value.getValue().intValue() + 1];
        NBErrors.requireNoErr(NBDeviceGetManufacturerA(getHandle(), id, new NBUInt(id.length), value));
        return Native.toString(id);
    }

    public final String getModel() {
        NBUIntByReference value = new NBUIntByReference();
        NBErrors.requireNoErr(NBDeviceGetModelA(getHandle(), null, new NBUInt(0L), value));
        byte[] id = new byte[value.getValue().intValue() + 1];
        NBErrors.requireNoErr(NBDeviceGetModelA(getHandle(), id, new NBUInt(id.length), value));
        return Native.toString(id);
    }

    public final String getSerialNumber() {
        NBUIntByReference value = new NBUIntByReference();
        NBErrors.requireNoErr(NBDeviceGetSerialNumberA(getHandle(), null, new NBUInt(0L), value));
        byte[] id = new byte[value.getValue().intValue() + 1];
        NBErrors.requireNoErr(NBDeviceGetSerialNumberA(getHandle(), id, new NBUInt(id.length), value));
        return Native.toString(id);
    }

    public final String getModuleSerialNumber() {
        NBUIntByReference value = new NBUIntByReference();
        NBErrors.requireNoErr(NBDeviceGetModuleSerialNumberA(getHandle(), null, new NBUInt(0L), value));
        byte[] id = new byte[value.getValue().intValue() + 1];
        NBErrors.requireNoErr(NBDeviceGetModuleSerialNumberA(getHandle(), id, new NBUInt(id.length), value));
        return Native.toString(id);
    }

    public final String getProduct() {
        NBUIntByReference value = new NBUIntByReference();
        NBErrors.requireNoErr(NBDeviceGetProductA(getHandle(), null, new NBUInt(0L), value));
        byte[] id = new byte[value.getValue().intValue() + 1];
        NBErrors.requireNoErr(NBDeviceGetProductA(getHandle(), id, new NBUInt(id.length), value));
        return Native.toString(id);
    }

    public final NBDeviceScanFormatInfo[] getSupportedScanFormats() {
        /* 298 */ NBUIntByReference uiCount = new NBUIntByReference();
        /* 299 */ NBErrors.requireNoErr(NBDeviceGetSupportedScanFormats(getHandle(), null, new NBUInt(0L), uiCount));
        /* 300 */ if (uiCount.getValue().intValue() == 0)
            return new NBDeviceScanFormatInfo[0];
        /* 301 */ int[] formats = new int[uiCount.getValue().intValue()];
        /* 302 */ NBErrors.requireNoErr(
                NBDeviceGetSupportedScanFormats(getHandle(), formats, new NBUInt(formats.length), uiCount));
        /* 303 */ int count = Math.min(formats.length, uiCount.getValue().intValue());
        /* 304 */ NBDeviceScanFormatInfo[] formatInfos = new NBDeviceScanFormatInfo[count];
        /* 305 */ for (int i = 0; i < count; i++) {
            /* 306 */ formatInfos[i] = getScanFormatInfo(NBDeviceScanFormat.get(formats[i]));
        }
        /* 308 */ return formatInfos;
    }

    public final boolean isScanRunning() {
        IntByReference value = new IntByReference();
        NBErrors.requireNoErr(NBDeviceIsScanRunning(getHandle(), value));
        return value.getValue() != 0;
    }

    public final void reset() {
        /* 332 */ NBErrors.requireNoErr(NBDeviceReset(getHandle()));
    }

    public boolean isScanFormatSupported(NBDeviceScanFormat scanFormat) {
        /* 342 */ IntByReference value = new IntByReference();
        /* 343 */ NBErrors.requireNoErr(NBDeviceIsScanFormatSupported(getHandle(), scanFormat.getValue(), value));
        /* 344 */ return value.getValue() != 0;
    }

    public NBDeviceScanFormatInfo getScanFormatInfo(NBDeviceScanFormat scanFormat) {
        /* 354 */ NBDeviceScanFormatInfoData.ByReference sScanFormatInfo = new NBDeviceScanFormatInfoData.ByReference();
        /* 355 */ NBErrors.requireNoErr(NBDeviceGetScanFormatInfo(getHandle(), scanFormat.getValue(), sScanFormatInfo));
        /* 356 */ return new NBDeviceScanFormatInfo(sScanFormatInfo);
    }

    public final int getParameter(long parameter) {
        /* 368 */ IntByReference value = new IntByReference();
        /* 369 */ NBErrors.requireNoErr(NBDeviceGetParameter(getHandle(), (int) parameter, value));
        /* 370 */ return value.getValue();
    }

    public final void setParameter(long parameter, int value) {
        /* 383 */ NBErrors.requireNoErr(NBDeviceSetParameter(getHandle(), (int) parameter, value));
    }

    public final int getFingerDetectValue() {
        /* 395 */ return getFingerDetectValue(DEFAULT_FINGER_DETECT_TYPE);
    }

    public final int getFingerDetectValue(NBDeviceFingerDetectType detectType) {
        /* 408 */ IntByReference value = new IntByReference();
        /* 409 */ NBErrors.requireNoErr(NBDeviceGetFingerDetectValue(getHandle(), detectType.getValue(), value));
        /* 410 */ return value.getValue();
    }

    public final void cancelScan() {
        NBErrors.requireNoErr(NBDeviceCancelScan(getHandle()));
    }

    public final NBDeviceScanResult scan() {
        /* 428 */ return scan(getSupportedScanFormats()[0]);
    }

    public final NBDeviceScanResult scan(NBDeviceScanFormat scanFormat) {
        /* 440 */ return scan(scanFormat, 0);
    }

    public final NBDeviceScanResult scan(NBDeviceScanFormat scanFormat, int flags) {
        /* 453 */ return scan(getScanFormatInfo(scanFormat), flags);
    }

    public final NBDeviceScanResult scan(NBDeviceScanFormatInfo scanFormatInfo) {
        /* 465 */ return scan(scanFormatInfo, 0L);
    }

    public final NBDeviceScanResult scan(NBDeviceScanFormatInfo scanFormatInfo, long flags) {
        /* 478 */ if (scanFormatInfo == null)
            throw new NullPointerException("scanFormatInfo");
        /* 479 */ int imageSize = scanFormatInfo.getWidth() * scanFormatInfo.getHeight();
        /* 480 */ byte[] image = new byte[imageSize];
        /* 481 */ IntByReference eStatus = new IntByReference();
        /* 482 */ NBErrors.requireNoErr(NBDeviceScan(getHandle(), scanFormatInfo.getFormat().getValue(), image,
                new NBSizeType(imageSize), (int) flags, eStatus));
        /* 483 */ return new NBDeviceScanResult(NBDeviceScanStatus.get(eStatus.getValue()), scanFormatInfo, image);
    }

    public final NBDeviceScanResult scanEx() {
        /* 494 */ return scanEx(-1);
    }

    public final NBDeviceScanResult scanEx(int timeout) {
        /* 507 */ return scanEx(timeout, null);
    }

    public NBDeviceScanResult scanEx(int timeout, NBDeviceScanPreviewListener previewListener) {
        /* 521 */ return scanEx(getSupportedScanFormats()[0], timeout, previewListener);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormat scanFormat) {
        /* 534 */ return scanEx(scanFormat, -1);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormatInfo scanFormatInfo) {
        /* 547 */ return scanEx(scanFormatInfo, -1);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormat scanFormat, NBDeviceScanPreviewListener previewListener) {
        /* 561 */ return scanEx(scanFormat, -1, previewListener);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormatInfo scanFormatInfo,
            NBDeviceScanPreviewListener previewListener) {
        /* 575 */ return scanEx(scanFormatInfo, -1, previewListener);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormat scanFormat, int timeout) {
        /* 590 */ return scanEx(scanFormat, timeout, null);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormatInfo scanFormatInfo, int timeout) {
        /* 605 */ return scanEx(scanFormatInfo, timeout, null);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormat scanFormat, int timeout,
            NBDeviceScanPreviewListener previewListener) {
        /* 621 */ return scanEx(scanFormat, timeout, previewListener, 0L);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormat scanFormat, int timeout,
            NBDeviceScanPreviewListener previewListener, long flags) {
        /* 638 */ return scanEx(getScanFormatInfo(scanFormat), timeout, previewListener, flags);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormatInfo scanFormatInfo, int timeout,
            NBDeviceScanPreviewListener previewListener) {
        /* 654 */ return scanEx(scanFormatInfo, timeout, previewListener, 0L);
    }

    public final NBDeviceScanResult scanEx(NBDeviceScanFormatInfo scanFormatInfo, int timeout,
            NBDeviceScanPreviewListener previewListener, long flags) {
        /* 671 */ if (scanFormatInfo == null)
            throw new NullPointerException("scanFormatInfo");
        /* 672 */ int imageSize = scanFormatInfo.getWidth() * scanFormatInfo.getHeight();
        /* 673 */ byte[] image = new byte[imageSize];
        /* 674 */ int id = NBCallbacks.add(this, previewListener);
        try {
            /* 676 */ IntByReference eStatus = new IntByReference();
            /* 677 */ NBErrors.requireNoErr(NBDeviceScanEx(getHandle(), scanFormatInfo.getFormat().getValue(), timeout,
                    previewListener != null ? NBDeviceScanPreviewCallback.previewCallback : null, new NBSizeType(id),
                    image, new NBSizeType(imageSize), (int) flags, eStatus));
            /* 678 */ return new NBDeviceScanResult(NBDeviceScanStatus.get(eStatus.getValue()), scanFormatInfo, image);
        } finally {
            /* 681 */ NBCallbacks.remove(id);
        }
    }

    public final byte[] ConvertImage(byte[] image, int width, int height, int dataDpi, NBDeviceEncodeFormat format,
            float compressionRatio, NBDeviceFingerPosition fingerPosition, int flags) {
        /* 702 */ if (image == null)
            throw new NullPointerException("image");
        /* 703 */ PointerByReference data = new PointerByReference();
        /* 704 */ IntByReference size = new IntByReference();
        try {
            /* 706 */ NBErrors.requireNoErr(NBDeviceConvertImage(image, width, height, dataDpi, format.getValue(),
                    compressionRatio, fingerPosition.getValue(), flags, data, size));
            /* 707 */ return data.getValue().getByteArray(0L, size.getValue());
        } finally {
            /* 710 */ NBErrors.requireNoErr(NBDeviceFree(data.getValue()));
        }
    }

    public static final int GetImageQuality(byte[] image, int width, int height, int dataDpi,
            NBDeviceImageQualityAlgorithm algorithm) {
        /* 725 */ if (image == null)
            throw new NullPointerException("image");
        /* 726 */ IntByReference quality = new IntByReference();
        /* 727 */ NBErrors
                .requireNoErr(NBDeviceImageQuality(image, width, height, dataDpi, algorithm.getValue(), quality));
        /* 728 */ return quality.getValue();
    }

    public final void SetBlobParameter(int uiParameter, byte[] pBlob) {
        /* 738 */ NBErrors.requireNoErr(
                NBDeviceSetBlobParameter(getHandle(), uiParameter, pBlob, pBlob == null ? 0 : pBlob.length));
    }

    public final byte[] GetBlobParameter(int uiParameter) {
        /* 748 */ PointerByReference data = new PointerByReference();
        /* 749 */ IntByReference size = new IntByReference();
        /* 750 */ byte[] res = null;
        try {
            /* 752 */ NBErrors.requireNoErr(NBDeviceGetBlobParameter(getHandle(), uiParameter, data, size));
            /* 753 */ res = data.getValue().getByteArray(0L, size.getValue());
        } finally {
            /* 756 */ NBErrors.requireNoErr(NBDeviceFree(data.getValue()));
        }
        /* 758 */ return res;
    }

    public final byte[] GenerateCalibrationData() {
        /* 767 */ PointerByReference data = new PointerByReference();
        /* 768 */ IntByReference size = new IntByReference();
        /* 769 */ byte[] res = null;
        try {
            /* 771 */ NBErrors.requireNoErr(NBDeviceGenerateCalibrationData(getHandle(), data, size));
            /* 772 */ res = data.getValue().getByteArray(0L, size.getValue());
        } finally {
            /* 775 */ NBErrors.requireNoErr(NBDeviceFree(data.getValue()));
        }
        /* 777 */ return res;
    }

    public final void enterStopMode(NBDeviceStopMode stopMode) {
        /* 790 */ enterStopMode(stopMode, 0);
    }

    public final void enterStopMode(NBDeviceStopMode stopMode, int fingerDetectionCount) {
        /* 804 */ NBErrors.requireNoErr(NBDeviceEnterStopMode(getHandle(), stopMode.getValue(), fingerDetectionCount));
    }

    public final void setLedState(NBDeviceLedState ledState) {
        /* 815 */ NBErrors.requireNoErr(NBDeviceSetLedState(getHandle(), ledState.getValue()));
    }

    public String toString() {
        /* 820 */ return String.format("%s %s (%s)", new Object[] { getManufacturer(), getModel(), getSerialNumber() });
    }

    public boolean equals(Object o) {
        /* 825 */ if (o == this)
            return true;
        /* 826 */ if (!(o instanceof NBDevice)) {
            /* 827 */ return false;
        }
        /* 829 */ NBDevice device = (NBDevice) o;
        /* 830 */ return (Pointer.nativeValue(device.getHandle()) == Pointer.nativeValue(getHandle()))
                && (device.getId().equals(getId()));
    }

    public int hashCode() {
        /* 835 */ long hashCode = 13L;
        /* 836 */ hashCode = hashCode * 397L ^ Pointer.nativeValue(getHandle());
        /* 837 */ String id = getId();
        /* 838 */ long idHashCode = "".equals(id) ? 0L : id.hashCode();
        /* 839 */ hashCode = hashCode * 397L ^ idHashCode;
        /* 840 */ return (int) hashCode;
    }

    public static final NBDevice connectToSpi(String spiName, int awakePin, int resetPin) {
        /* 858 */ return connectToSpi(spiName, awakePin, resetPin, 0L);
    }

    public static final NBDevice connectToSpi(String spiName, int awakePin, int resetPin, long flags) {
        /* 873 */ PointerByReference hHandle = new PointerByReference();
        /* 874 */ NBErrors.requireNoErr(NBDeviceConnectToSpiA(spiName, awakePin, resetPin, (int) flags, hHandle));
        /* 875 */ return new NBDevice(hHandle.getValue());
    }

    public static final NBDevice connectToCustom(NBUCommInterface commInterface, long flags) {
        /* 888 */ PointerByReference hHandle = new PointerByReference();
        /* 889 */ NBErrors
                .requireNoErr(NBDeviceConnectToCustom(commInterface.getCommInterfaceStruct(), (int) flags, hHandle));
        /* 890 */ return new NBDevice(hHandle.getValue());
    }

    public static final NBDevice connectToBluetooth(long flags) {
        PointerByReference hHandle = new PointerByReference();
        NBUCommInterface commInterface = new BluetoothCommInterface();
        NBErrors.requireNoErr(NBDeviceConnectToCustom(commInterface.getCommInterfaceStruct(), (int) flags, hHandle));
        return new NBDevice(hHandle.getValue());
    }

    public static final NBDevice connectToSpi(String spiName, int awakePin, int resetPin, int chipSelectPin) {
        return connectToSpi(spiName, awakePin, resetPin, chipSelectPin, 0L);
    }

    public static final NBDevice connectToSpi(String spiName, int awakePin, int resetPin, int chipSelectPin,
            long flags) {
        PointerByReference hHandle = new PointerByReference();
        NBErrors.requireNoErr(
                NBDeviceConnectToSpiExA(spiName, awakePin, resetPin, chipSelectPin, (int) flags, hHandle));
        return new NBDevice(hHandle.getValue());
    }

    public final void openSession(byte[] keyId, byte[] key) {
        NBErrors.requireNoErr(
                NBDeviceOpenSession(getHandle(), keyId, new NBSizeType(keyId.length), key, new NBSizeType(key.length)));
    }

    public final void closeSession() {
        NBErrors.requireNoErr(NBDeviceCloseSession(getHandle()));
    }

    public final boolean isSessionOpen() {
        IntByReference value = new IntByReference();
        NBErrors.requireNoErr(NBDeviceIsSessionOpen(getHandle(), value));
        return value.getValue() != 0;
    }

    public final void setCustomScanFormat(int horizontalOrigin, int verticalOrigin, int width, int height,
            int horizontalResolution, int verticalResolution) {
        NBErrors.requireNoErr(NBDeviceSetCustomScanFormat(getHandle(), horizontalOrigin, verticalOrigin, width, height,
                horizontalResolution, verticalResolution));
    }

    public final NBDeviceCapabilities getCapabilities() {
        /* 985 */ PointerByReference data = new PointerByReference();
        /* 986 */ NBErrors.requireNoErr(NBDeviceGetCapabilities(getHandle(), data));
        /* 987 */ NBDeviceCapabilities capabilities = new NBDeviceCapabilities(data.getValue().getByteArray(0L, 64));
        /* 988 */ NBErrors.requireNoErr(NBDeviceFree(data.getValue()));
        /* 989 */ return capabilities;
    }

    public static boolean SupportsNBUApi(int vid, int pid) {
        /* 999 */ IntByReference value = new IntByReference();
        /* 1000 */ NBErrors.requireNoErr(NBDeviceSupportsNBUApi(vid, pid, value));
        /* 1001 */ return value.getValue() != 0;
    }

    protected void dispose(boolean disposing) {
        /* 1014 */ if ((getOwnsHandle()) && (getHandle() != null))
            /* 1015 */ NBErrors.requireNoErr(NBDeviceDestroy(getHandle()));
    }

    static {
        /* 122 */ Native.register(NBDevice.class, NBDevicesLibrary.NATIVE_LIBRARY);
    }
}
